<?php
session_start();
include('adm/bd/db.php'); // Conexión a la base de datos

if (!isset($_SESSION['cliente_id'])) {
    header('Location: index.php'); // Redirigir si no hay cliente registrado
    exit();
}

// Variables del cliente
$cliente_id = $_SESSION['cliente_id'];
$nombre_cliente = $_SESSION['nombre'];
$telefono_cliente = $_SESSION['telefono']; // Se asume que el teléfono está en sesión

// Validar si el cliente puede realizar un nuevo pedido
$sql_validar_tiempo = "SELECT id, TIMESTAMPDIFF(HOUR, MAX(fecha_hora), NOW()) AS horas_desde_ultimo_pedido
                       FROM pedidos
                       WHERE cliente_id = ?
                       GROUP BY cliente_id";
$stmt_validar_tiempo = $conn->prepare($sql_validar_tiempo);
$stmt_validar_tiempo->bind_param("i", $cliente_id);
$stmt_validar_tiempo->execute();
$result_validar_tiempo = $stmt_validar_tiempo->get_result();
$row_validar_tiempo = $result_validar_tiempo->fetch_assoc();
$horas_desde_ultimo_pedido = $row_validar_tiempo['horas_desde_ultimo_pedido'] ?? 25;

// Insertar pedido si se envía el formulario y está permitido
if ($_SERVER['REQUEST_METHOD'] == 'POST' && $horas_desde_ultimo_pedido >= 24) {
    $tema = !empty($_POST['tema']) ? htmlspecialchars($_POST['tema']) : '';
    $detalles = !empty($_POST['detalles']) ? htmlspecialchars($_POST['detalles']) : '';

    if (empty($tema) && empty($detalles)) {
        $mensaje = "Debes rellenar al menos uno de los campos.";
    } else {
        $sql_pedido = "INSERT INTO pedidos (cliente_id, tema, detalles, categoria, estado, fecha_hora) 
                       VALUES (?, ?, ?, 'Retro', 'Pendiente', NOW())";
        $stmt_pedido = $conn->prepare($sql_pedido);
        $stmt_pedido->bind_param("iss", $cliente_id, $tema, $detalles);

        if ($stmt_pedido->execute()) {
            $mensaje = "Tu pedido musical fue registrado correctamente.";
        } else {
            $mensaje = "Error al registrar el pedido.";
        }
    }
} elseif ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $mensaje = "Solo puedes pedir un tema cada 24 horas.";
}

// Consultar el tema más reciente del cliente actual
$sql_mi_tema = "
    SELECT p.id, 
           IF(m.tema IS NOT NULL, m.tema, p.detalles) AS tema_mostrado,
           p.estado,
           p.tema IS NULL AS es_especial
    FROM pedidos p
    LEFT JOIN musica m ON p.tema = m.tema
    WHERE p.cliente_id = ?
    ORDER BY p.fecha_hora DESC
    LIMIT 1";
$stmt_mi_tema = $conn->prepare($sql_mi_tema);
$stmt_mi_tema->bind_param("i", $cliente_id);
$stmt_mi_tema->execute();
$result_mi_tema = $stmt_mi_tema->get_result();
$mi_tema = $result_mi_tema->fetch_assoc();

// Consultar los últimos 5 temas de otros clientes
$sql_otros_temas = "
    SELECT c.nombre, 
           IF(m.tema IS NOT NULL, m.tema, p.detalles) AS tema_mostrado,
           p.estado,
           p.tema IS NULL AS es_especial
    FROM pedidos p
    LEFT JOIN musica m ON p.tema = m.tema
    JOIN clientes c ON p.cliente_id = c.id
    WHERE p.cliente_id != ?
    ORDER BY p.fecha_hora DESC
    LIMIT 5";
$stmt_otros_temas = $conn->prepare($sql_otros_temas);
$stmt_otros_temas->bind_param("i", $cliente_id);
$stmt_otros_temas->execute();
$result_otros_temas = $stmt_otros_temas->get_result();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Pedidos - Retro</title>
    <style>
        body {
            margin: 0;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-image: url('img/fo2.jpeg');
            background-size: cover;
            background-position: center;
            background-attachment: fixed;
            color: #333;
        }
        .content-container {
            margin: 20px auto;
            width: 80%;
            max-width: 800px;
            background: rgba(255, 255, 255, 0.9);
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            text-align: center;
        }
        h2, h3 {
            color: #0078D7;
        }
        form {
            margin-bottom: 20px;
            display: flex;
            flex-direction: column;
            align-items: center;
        }
        input, textarea {
            margin: 10px 0;
            padding: 10px;
            width: 90%;
            max-width: 600px;
            border-radius: 5px;
            border: 1px solid #ccc;
        }
        #tema {
            font-size: 16px;
            padding: 12px;
            width: 100%;
        }
        button {
            background-color: #0078D7;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 20px;
            font-weight: bold;
            cursor: pointer;
            margin-top: 10px;
        }
        button:hover {
            background-color: #005BB5;
        }
        button:disabled {
            background-color: #ddd;
            cursor: not-allowed;
        }
        .tema-container {
            margin: 20px auto;
            padding: 20px;
            background: #f9f9f9;
            border: 1px solid #ddd;
            border-radius: 15px;
            text-align: center;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        .tema-container h3 {
            font-size: 20px;
            font-weight: bold;
            text-transform: uppercase;
            margin-bottom: 20px;
            color: #444;
        }
        .tema-container .detalle {
            font-size: 18px;
            font-weight: bold;
            margin: 10px 0;
        }
        .tema-container .estado {
            font-size: 16px;
            font-weight: bold;
            margin-top: 10px;
        }
        .estado-tocado {
            color: salmon;
        }
        .estado-aceptado {
            color: green;
        }
        .estado-no-se-toca {
            color: red;
        }
        .estado-pendiente {
            color: gray;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }
        th, td {
            padding: 10px;
            text-align: left;
            border: 1px solid #ddd;
        }
        th {
            background-color: #0078D7;
            color: white;
        }
        .warning {
            color: red;
            font-weight: bold;
            margin-top: 10px;
        }
        #resultados {
            position: absolute;
            background: white;
            border: 1px solid #ccc;
            width: 100%;
            max-height: 200px;
            overflow-y: auto;
            display: none;
            z-index: 100;
        }
        .resultado {
            padding: 5px;
            cursor: pointer;
            border-bottom: 1px solid #ddd;
        }
        .resultado:hover {
            background-color: #f0f0f0;
        }
    </style>
</head>
<body>
    <div class="content-container">
        <h2>Hola, <?php echo htmlspecialchars($nombre_cliente); ?> 👋</h2>

        <!-- Formulario para solicitar tema -->
        <form action="" method="post">
            <div style="position: relative;">
                <input type="text" id="tema" name="tema" placeholder="&#127911; Buscar canción...">
                <div id="resultados"></div>
            </div>
            <textarea name="detalles" rows="4" placeholder="&#128221; Si no encuentras la canción, ingresala aquí (opcional)"></textarea>
            <button type="submit" <?php echo $horas_desde_ultimo_pedido < 24 ? 'disabled' : ''; ?>>SOLICITAR CANCIÓN</button>
            
            <?php if ($horas_desde_ultimo_pedido < 24): ?>
                <p class="warning">⚠️ Solo puedes pedir una canción por noche. ⚠️</p>
            <?php endif; ?>
        </form>

        <!-- Tema del cliente -->
        <?php if ($mi_tema): ?>
            <div class="tema-container">
                <h3>Tu Pedido</h3>
                <p class="detalle">Canción:</p>
                <p style="<?php echo $mi_tema['es_especial'] ? 'color: red;' : ''; ?>">
                    <?php echo htmlspecialchars($mi_tema['tema_mostrado']); ?>
                </p>
                <p class="detalle">Estado:</p>
                <p class="estado <?php echo strtolower(str_replace(' ', '-', "estado-" . $mi_tema['estado'])); ?>">
                    <?php 
                    switch ($mi_tema['estado']) {
                        case 'Tocado':
                            echo "Tu petición ya fue reproducida";
                            break;
                        case 'Aceptado':
                            echo "¡Recibimos tu petición y la estamos Bucando!";
                            break;
                        case 'No se toca':
                            echo "Disculpa pero el tema que solicitaste no es compatible con el formto de esta fiesta.";
                            break;
                        default:
                            echo "Recibido por el Dj";
                    }
                    ?>
                </p>
            </div>
        <?php endif; ?>

        <!-- Últimos pedidos de otros clientes -->
        <h3>Últimos 5 Pedidos de Otros Clientes</h3>
        <table>
            <thead>
                <tr>
                    <th>USUARIO</th>
                    <th>ARTISTA - CANCIÓN</th>
                    <th>ESTADO</th>
                </tr>
            </thead>
            <tbody>
                <?php if ($result_otros_temas->num_rows > 0): ?>
                    <?php while ($row = $result_otros_temas->fetch_assoc()): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($row['nombre']); ?></td>
                            <td style="<?php echo $row['es_especial'] ? 'color: red;' : ''; ?>">
                                <?php echo htmlspecialchars($row['tema_mostrado']); ?>
                            </td>
                            <td class="<?php echo strtolower(str_replace(' ', '-', "estado-" . $row['estado'])); ?>">
                                <?php 
                                switch ($row['estado']) {
                                    case 'Tocado':
                                        echo "Reproducida";
                                        break;
                                    case 'Aceptado':
                                        echo "Próximamente";
                                        break;
                                    case 'No se toca':
                                        echo "Incompatible";
                                        break;
                                    default:
                                        echo "En espera";
                                }
                                ?>
                            </td>
                        </tr>
                    <?php endwhile; ?>
                <?php else: ?>
                    <tr><td colspan="3">No hay pedidos recientes.</td></tr>
                <?php endif; ?>
            </tbody>
        </table>
    </div>

    <!-- Scripts -->
    <script>
        const actualizarEstado = () => {
            fetch(`estado_pedido.php?cliente_id=<?php echo $cliente_id; ?>`)
                .then(response => response.json())
                .then(data => {
                    if (data && data.estado) {
                        const estadoSpan = document.querySelector('.estado');
                        estadoSpan.textContent = obtenerMensajeEstado(data.estado);
                        estadoSpan.className = `estado estado-${data.estado.toLowerCase().replace(/\s/g, '-')}`;
                    }
                })
                .catch(error => console.error('Error actualizando estado:', error));
        };

        const obtenerMensajeEstado = (estado) => {
            switch (estado) {
                case 'Tocado':
                    return 'Tu petición ya fue reproducida';
                case 'Aceptado':
                    return '¡Recibimos tu petición y la estamos Bucando!';
                case 'No se toca':
                    return 'Disculpa pero el tema que solicitaste no es compatible con el formto de esta fiesta.';
                default:
                    return 'Recibido por el Dj';
            }
        };

        const buscarTemas = () => {
            const temaInput = document.getElementById('tema');
            temaInput.addEventListener('input', () => {
                const query = temaInput.value;
                if (query.length > 1) {
                    fetch(`buscar.php?q=${encodeURIComponent(query)}`)
                        .then(response => response.json())
                        .then(data => {
                            const resultados = document.getElementById('resultados');
                            resultados.innerHTML = '';
                            resultados.style.display = 'block';

                            if (data.length > 0) {
                                data.forEach(item => {
                                    const div = document.createElement('div');
                                    div.textContent = `${item.tema} - ${item.artista}`;
                                    div.className = 'resultado';
                                    div.addEventListener('click', () => {
                                        temaInput.value = item.tema;
                                        resultados.style.display = 'none';
                                    });
                                    resultados.appendChild(div);
                                });
                            } else {
                                resultados.innerHTML = '<div class="resultado">No se encontraron resultados</div>';
                            }
                        })
                        .catch(error => console.error('Error en el buscador:', error));
                } else {
                    document.getElementById('resultados').style.display = 'none';
                }
            });
        };

        setInterval(actualizarEstado, 5000);
        buscarTemas();
    </script>
</body>
</html>
