<?php
// Habilitar la visualización de errores (solo para desarrollo)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Iniciar sesión y conectar a la base de datos
session_start();
include('bd/db.php'); // Asegúrate de que este archivo establece la conexión y asigna a $conn

// Validar sesión de administrador para acceder a funcionalidades críticas
$isAdmin = isset($_SESSION['admin']);

// Función para borrar todo el chat (solo para administradores)
function borrar_chat($conn) {
    // Preparar y ejecutar la consulta para borrar todos los mensajes
    $sql_borrar = "DELETE FROM chat";
    if ($conn->query($sql_borrar) === TRUE) {
        return true;
    } else {
        return false;
    }
}

// Exportar chat a Excel
if (isset($_GET['exportar']) && $_GET['exportar'] === 'excel') {
    header('Content-Type: application/vnd.ms-excel');
    header('Content-Disposition: attachment; filename="chat_historial.xls"');
    header('Pragma: no-cache');
    header('Expires: 0');

    $sql_exportar = "SELECT nombre, mensaje, fecha_envio, leido FROM chat ORDER BY fecha_envio ASC";
    $result_exportar = $conn->query($sql_exportar);

    if ($result_exportar) {
        echo "Nombre\tMensaje\tFecha de Envío\tLeído\n";
        while ($row = $result_exportar->fetch_assoc()) {
            // Reemplazar tabulaciones y saltos de línea para evitar problemas en Excel
            $nombre = str_replace(["\t", "\n", "\r"], ' ', $row['nombre']);
            $mensaje = str_replace(["\t", "\n", "\r"], ' ', $row['mensaje']);
            echo "{$nombre}\t{$mensaje}\t{$row['fecha_envio']}\t" . ($row['leido'] ? 'Sí' : 'No') . "\n";
        }
    } else {
        echo "Error al exportar chat: " . $conn->error;
    }
    exit();
}

// Borrar todo el chat (solo para administradores)
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['borrar_chat'])) {
    if ($isAdmin) {
        if (borrar_chat($conn)) {
            // Borrado exitoso, redirigir para actualizar la vista
            header('Location: ' . $_SERVER['PHP_SELF']);
            exit();
        } else {
            die("Error al borrar el chat: " . $conn->error);
        }
    } else {
        die("Acción no autorizada.");
    }
}

// Insertar un nuevo mensaje
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['mensaje']) && !isset($_POST['borrar_chat'])) {
    // Obtener y sanitizar los datos del formulario
    $nombre = trim($_POST['nombre']);
    $mensaje = trim($_POST['mensaje']);

    if (!empty($nombre) && !empty($mensaje)) {
        // Limitar la longitud del nombre y mensaje para mayor seguridad
        if (strlen($nombre) > 100) {
            die("El nombre excede la longitud permitida.");
        }
        if (strlen($mensaje) > 1000) {
            die("El mensaje excede la longitud permitida.");
        }

        // Preparar la consulta de inserción
        $sql_insert = "INSERT INTO chat (nombre, mensaje) VALUES (?, ?)";
        $stmt = $conn->prepare($sql_insert);
        if ($stmt) {
            // Sanitizar el nombre y mensaje para evitar inyecciones
            $nombre_sanitizado = htmlspecialchars($nombre, ENT_QUOTES, 'UTF-8');
            $mensaje_sanitizado = htmlspecialchars($mensaje, ENT_QUOTES, 'UTF-8');

            $stmt->bind_param('ss', $nombre_sanitizado, $mensaje_sanitizado);
            if ($stmt->execute()) {
                // Inserción exitosa, redirigir para evitar reenvío del formulario
                header('Location: ' . $_SERVER['PHP_SELF']);
                exit();
            } else {
                die("Error al insertar mensaje: " . $stmt->error);
            }
            $stmt->close();
        } else {
            die("Error en la preparación de la consulta: " . $conn->error);
        }
    } else {
        die("Nombre o mensaje vacío.");
    }
}

// Obtener los últimos mensajes en orden ascendente para que los más antiguos estén arriba
$sql_mensajes = "SELECT nombre, mensaje, fecha_envio, leido FROM chat ORDER BY fecha_envio ASC LIMIT 50";
$result_mensajes = $conn->query($sql_mensajes);

if (!$result_mensajes) {
    die("Error en la consulta SQL de mensajes: " . $conn->error);
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Chat Interno</title>
    <style>
        /* Reset de márgenes y padding */
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        /* Estilos del cuerpo y html para ocupar toda la altura */
        html, body {
            height: 100%;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: #f3f4f6;
            color: #333;
        }

        /* Cabecera fija */
        .cabecera {
            position: fixed;
            top: 0;
            width: 100%;
            height: 60px; /* Ajusta según tu diseño */
            background: #0078D7;
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 1000; /* Por encima de otros elementos */
        }

        /* Contenido principal con margen superior para la cabecera */
        .main-content {
            margin-top: 60px; /* Igual a la altura de la cabecera */
            height: calc(100% - 60px);
            display: flex;
            justify-content: center;
            align-items: center;
            padding: 20px;
        }

        /* Contenedor del chat */
        .content-container {
            width: 100%;
            max-width: 800px;
            background: white;
            border-radius: 10px;
            padding: 20px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            display: flex;
            flex-direction: column;
            height: 100%;
        }

        /* Título del chat */
        h2 {
            text-align: center;
            color: #0078D7;
            margin-bottom: 20px;
        }

        /* Área del chat con scroll */
        .chat-box {
            flex: 1; /* Ocupa todo el espacio disponible */
            background: #f9fafc;
            border: 1px solid #ddd;
            border-radius: 5px;
            padding: 10px;
            overflow-y: auto; /* Scroll vertical */
            margin-bottom: 20px;
        }

        /* Estilos para cada mensaje */
        .chat-message {
            margin-bottom: 10px;
            padding: 8px;
            border-radius: 5px;
            background: #e9f5ff;
        }

        .chat-message strong {
            display: block;
            color: #0078D7;
            margin-bottom: 5px;
        }

        .chat-message p {
            margin-bottom: 5px;
        }

        .chat-message span {
            font-size: 12px;
            color: #999;
        }

        /* Formulario de envío de mensajes */
        .chat-footer {
            display: flex;
            flex-direction: column;
            gap: 10px;
        }

        .chat-footer input, .chat-footer textarea {
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 5px;
            width: 100%;
            resize: none; /* Evita el cambio de tamaño del textarea */
        }

        .chat-footer textarea {
            height: 100px;
        }

        .chat-footer button {
            background: #0078D7;
            color: white;
            border: none;
            padding: 10px;
            border-radius: 5px;
            cursor: pointer;
            transition: background 0.3s ease;
        }

        .chat-footer button:hover {
            background: #005BB5;
        }

        /* Botón de exportar */
        .export-button {
            text-align: center;
            margin-top: 20px;
        }

        .export-button a {
            text-decoration: none;
            background: #28a745;
            color: white;
            padding: 10px 20px;
            border-radius: 5px;
            transition: background 0.3s ease;
        }

        .export-button a:hover {
            background: #218838;
        }

        /* Botón para borrar todo el chat */
        .delete-button {
            text-align: center;
            margin-top: 10px;
        }

        .delete-button form {
            display: inline-block;
        }

        .delete-button button {
            background: #dc3545;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 5px;
            cursor: pointer;
            transition: background 0.3s ease;
        }

        .delete-button button:hover {
            background: #c82333;
        }

        /* Responsive */
        @media (max-width: 600px) {
            .content-container {
                padding: 10px;
            }

            .chat-footer textarea {
                height: 80px;
            }
        }
    </style>
</head>
<body>
    <!-- Cabecera fija (menú) -->
    <div class="cabecera">
        <?php include('cabecera.php'); ?>
    </div>

    <!-- Contenido principal -->
    <div class="main-content">
        <div class="content-container">
            <h2>💬 Chat Interno</h2>

            <!-- Mostrar mensajes del chat -->
            <div class="chat-box" id="chat-box">
                <?php if ($result_mensajes->num_rows > 0): ?>
                    <?php while ($row = $result_mensajes->fetch_assoc()): ?>
                        <div class="chat-message">
                            <strong><?php echo htmlspecialchars($row['nombre']); ?></strong>
                            <p><?php echo nl2br(htmlspecialchars($row['mensaje'])); ?></p>
                            <span><?php echo date("d/m/Y H:i", strtotime($row['fecha_envio'])); ?> | <?php echo $row['leido'] ? '✅ Leído' : '📩 No leído'; ?></span>
                        </div>
                    <?php endwhile; ?>
                <?php else: ?>
                    <p>No hay mensajes en el chat.</p>
                <?php endif; ?>
            </div>

            <!-- Formulario para enviar mensaje -->
            <form class="chat-footer" method="POST">
                <input type="text" name="nombre" placeholder="Tu nombre" required>
                <textarea name="mensaje" placeholder="Escribe tu mensaje..." required></textarea>
                <button type="submit">Enviar</button>
            </form>

            <!-- Botón para exportar el chat a Excel -->
            <div class="export-button">
                <a href="?exportar=excel">📥 Exportar Chat a Excel</a>
            </div>

            <!-- Botón para borrar todo el chat (solo para administradores) -->
            <?php if ($isAdmin): ?>
                <div class="delete-button">
                    <form method="POST" onsubmit="return confirmarBorrado();">
                        <button type="submit" name="borrar_chat">🗑️ Borrar Todo el Chat</button>
                    </form>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Pie de página -->
    <?php include('pie.php'); ?>

    <!-- Script para scroll automático al final del chat y confirmación de borrado -->
    <script>
        // Función para desplazar el chat al final
        function scrollToBottom() {
            var chatBox = document.getElementById('chat-box');
            chatBox.scrollTop = chatBox.scrollHeight;
        }

        // Ejecutar al cargar la página
        window.onload = function() {
            scrollToBottom();
        };

        // Ejecutar después de enviar un mensaje
        window.onunload = function() {
            scrollToBottom();
        };

        // Función para confirmar el borrado del chat
        function confirmarBorrado() {
            return confirm('¿Estás seguro de que deseas borrar todo el chat? Esta acción no se puede deshacer.');
        }
    </script>
</body>
</html>
